package com.ejie.ab04b.service.platea;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.ejie.ab04b.constantes.ConstantesPlateaTramitacion;
import com.ejie.ab04b.exception.AB04BException;
import com.ejie.ab04b.parse.MessageParserSax;
import com.ejie.ab04b.util.seguridad.UtilSeguridad;
import com.ejie.client.fdcdp.Q99RCPFacadeWSSoap;
import com.ejie.sc.xml.ConsultationContext;

/**
 * Clase que implementa los metodos a utilizar para invocar a los servicios web
 * del FDCDP
 * 
 * @author gfi
 * 
 */
@Service()
public class FDCDPFacadeServiceImpl implements FDCDPFacadeService {

	private static final Logger LOGGER = LoggerFactory
			.getLogger(FDCDPFacadeServiceImpl.class);

	@Autowired()
	private Q99RCPFacadeWSSoap service;

	/**
	 * Servicio que recupera, dado un id de procedimiento, su informacion
	 * asociada en el FDCDP
	 * 
	 *  idProcedure
	 *            String
	 *  ConsultationContext ConsultationContext.
	 *
	 * @param idProcedure the id procedure
	 * @return the soft consultation context
	 * @throws AB04BException the AB 04 B exception
	 */
	public ConsultationContext getSoftConsultationContext(String idProcedure)
			throws AB04BException {
		try {
			String result = null;
			ConsultationContext consContext = new ConsultationContext();

			// Obtenemos el token de sesion:
			String token = UtilSeguridad.getInstance().getTokenXLNets();

			try {

				result = this.service.getSoftConsultationContext(token,
						idProcedure);
				consContext = this.parseRespuesta(result);
				FDCDPFacadeServiceImpl.LOGGER
						.info("Se ha obtenido la informacion del FDCDP del procedure "
								+ idProcedure);
				FDCDPFacadeServiceImpl.LOGGER.info("RESPUESTA: {}", result);
			} catch (Exception e) {
				FDCDPFacadeServiceImpl.LOGGER.error(
						"Error al obtener la informacion del FDCDP del procedure "
								+ idProcedure, e);
			}

			return consContext;
		} catch (Exception e) {
			throw new AB04BException(e, e.getMessage());
		}
	}

	/**
	 *  respuesta
	 *            String
	 *  ConsultationContext.
	 *
	 * @param respuesta the respuesta
	 * @return the consultation context
	 * @throws AB04BException the AB 04 B exception
	 */
	private ConsultationContext parseRespuesta(String respuesta)
			throws AB04BException {
		try {
			ConsultationContext consultationContext = null;

			// Parseamos la respuesta
			consultationContext = (ConsultationContext) new MessageParserSax<ConsultationContext>(
					ConsultationContext.class.getPackage().getName(),
					ConstantesPlateaTramitacion.NAMESPACE_PLATEA_GENERIC)
					.parse(respuesta);

			return consultationContext;
		} catch (Exception e) {
			throw new AB04BException(e, e.getMessage());
		}
	}
}